<?php

namespace App\Http\Controllers;

use App\Models\StakeholderReminder;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;

class StakeholderReminderController extends Controller
{
    public function index(Request $request)
    {
        $query = StakeholderReminder::query();

        if ($request->filled('status')) {
            $query->where('status', $request->input('status'));
        }
        if ($request->filled('reminder_to')) {
            $query->where('reminder_to', $request->input('reminder_to'));
        }
        if ($request->filled('month')) {
            $month = $request->input('month');
            $query->whereRaw("DATE_FORMAT(COALESCE(recorded_at, created_at), '%Y-%m') = ?", [$month]);
        }
        if ($request->filled('q')) {
            $q = trim($request->input('q'));
            $query->where(function ($qb) use ($q) {
                $qb->where('name', 'like', "%{$q}%")
                   ->orWhere('reminder_to', 'like', "%{$q}%")
                   ->orWhereJsonContains('reminder_emails', $q)
                   ->orWhere('status', 'like', "%{$q}%");
            });
        }

        $items = $query->orderByDesc('created_at')->paginate(10)->appends($request->query());
        return view('stakeholders.index', compact('items'));
    }

    public function create()
    {
        return view('stakeholders.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'target_date' => ['required', 'date'],
            'reminder_to' => ['required', 'in:Client,Internal PIC'],
            'reminder_frequency_hours' => ['required', 'in:4,8,12,24'],
            'reminder_emails' => ['required', 'array', 'min:1'],
            'reminder_emails.*' => ['email'],
        ]);

        $data['recorded_at'] = Carbon::now()->toDateString();
        $data['status'] = 'Sedang Dikerjakan';

        StakeholderReminder::create($data);

        return redirect()->route('stakeholders.index')->with('success', 'Reminder stakeholder ditambahkan.');
    }

    public function show(StakeholderReminder $stakeholder)
    {
        return view('stakeholders.show', ['item' => $stakeholder]);
    }

    public function edit(StakeholderReminder $stakeholder)
    {
        return view('stakeholders.edit', ['item' => $stakeholder]);
    }

    public function update(Request $request, StakeholderReminder $stakeholder)
    {
        $data = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'target_date' => ['required', 'date'],
            'status' => ['required', 'in:Sedang Dikerjakan,Selesai'],
            'reminder_to' => ['required', 'in:Client,Internal PIC'],
            'reminder_frequency_hours' => ['required', 'in:4,8,12,24'],
            'reminder_emails' => ['required', 'array', 'min:1'],
            'reminder_emails.*' => ['email'],
        ]);

        $stakeholder->update($data);

        return redirect()->route('stakeholders.index')->with('success', 'Reminder stakeholder diperbarui.');
    }

    public function destroy(StakeholderReminder $stakeholder)
    {
        $stakeholder->delete();
        return redirect()->route('stakeholders.index')->with('success', 'Reminder stakeholder dihapus.');
    }
}
